#ifndef EXPERIMENTER_PLUGIN_H
#define EXPERIMENTER_PLUGIN_H

// EXPERIMENTER_API_VER 1 corresponds to versions 1.3 and 1.4 of the released RatCog system.
// EXPERIMENTER_API_VER 2 corresponds to version 1.5 of RatCog.
#define EXPERIMENTER_API_VER 2

#include <OS.h>
#include <Message.h>
#include "Debug.h"
#include "UserEnvMessage.h"
#include "PortMessage.h"
#include "DatabaseMessage.h"
#include "WaitQueue.h"

extern DebugServer *GLOB_debug;

/*******************************************************
* Created: 7/21/00, CGP
*
* For each experiment there is both a rat brain plugin and
* an experimenter plugin. The rat brain controls the rat's behavior
* and the experimenter plugin controls the experimenter's behavior.
* That is, the experimenter plugin modifies the environment (maze)
* based on the rat's actions. The experimenter plugin also sets up the
* maze for each trial.
*
* Modified: CGP, 9/15/00, to enable plugin parameter setting.
*		For now, plugin parameter settings can only be established
*		when the plugin is created. The contents of the BMessage given
*		to the constructor (i.e., the number, type, and name of the data items)
*		is established by protocol agreement between the plugin designer and
*		plugin user (plugin users are RatCog users).
* Modified: CGP, 10/12/00, to add dbMsg parameter to constructor.
*		The experimenter needs acces to the database.
* Modified: CGP, 11/4/00; Started to create a unified experimenter API
* 		where all the calls appear on this interface instead of having
*		to go to the database, environment, and port interfaces.
* Modified: CGP, 12/21/00; Added some common functions for use by experimenters:
*		WaitForMessage, WaitForArmEntry
* Modified: CGP, 1/3/01; Removed code from .h file & put in .cpp file.
* Modified: CGP, 1/6/01
*		Experimenter API now is event driven. No message receive loop is now present in
*		experimenters. Instead, you redeclare and override the virtual functions. E.g.,
*			Setup, RunTrial, RatConsumedFood, and RatMoved.
*******************************************************/

class UserEnvMessage;

class ExperimenterPlugin {
   public:
   // Call constructor to set up the environment for running a new rat.
   // expRatPort enables sending messages to the rat's port.
   // envExpPort enables experimenter to receive port messages from environment.
   // userEnvMsg is the message (BLooper based) interface to the environment.
   
   // Experimenter constructor will retrieve settings from database. If no settings
   // specified in database, will use its internal default settings and save these
   // to the database.
	ExperimenterPlugin(PortMessage *expRatPort,
		PortMessage *envExpPort, DatabaseMessage *dbMsg, UserEnvMessage *userEnvMsg,
		DebugServer *bugServer);
	virtual ~ExperimenterPlugin();
	long Start(void); // internal function used to run thread for experimenter
	bool VersionOK(); // check for API version compatiblity of the loaded plugin
	// This function is used internally in the maze simulator.
	// Use this only before experimenter thread is started to get default parameters.
	BMessage *GetSettings(); // caller owns returned object

	// Use the implementation for this virtual function
	// that is used the other experimenter plugins (e.g., StupidExperimenter.h).
	virtual int PLUGIN_API_Version() { return -1; }

	//////////////////////////////////////////////////////////////////////////
	// To implement an experimenter, you redeclare and override the following
	// PLUGIN_ virtual functions.
	//////////////////////////////////////////////////////////////////////////

	// While a PLUGIN_ function is being executed in the experimenter
	// another PLUGIN_ function will *not* be called by ExperimenterPlugin.cpp.
	// A consequence of this
	// is that you can use the EXP_WaitForEvent call (see below) in your PLUGIN_
	// code and be assured of handling events with EXP_WaitForEvent and not with
	// PLUGIN_RatConsumedFood and PLUGIN_RatMoved. These latter two PLUGIN_ functions
	// will only be called to handle events when EXP_WaitForEvent is not used to handle
	// the events.

	// This is called once just as the experimenter is started. This is called after
	// the constructor for the plugin is used and after the threads for the experimenter
	// are started. All experimenter API functions can be used when coding Setup.
	// Since all API functions can be used in Setup, calls to RatConsumedFood and
	// and RatMoved may result indirectly from call to Setup because the rat can be
	// be placed on the maze in Setup. It is strongly advised that you take the rat
	// off the maze prior to completing Setup (if you indeed do put the rat on the maze
	// in Setup) sot hat RunTrial and the other functions can proceed with the rat
	// initiall off the maze.
	virtual void PLUGIN_Setup() {};
	
	// This is called to run a trial when the GUI user asks to run a trial.
	// The rat should be off the maze when this function finishes. (NOTE:
	// this is *not* called to complete a trial when the rat has been in single step
	// mode; it is only called to run a complete trial from start to end).
	virtual void PLUGIN_RunTrial() {};
	
	// This is called when GUI user asks to run the current rat on remaining trials.
	// RunTrial may have been called before hand, in which case the remaining trials
	// for the current rat are completed.
	// The rat should be off the maze when this function finishes.
	virtual void PLUGIN_RunCurrentRat() {};
	
	// Called when rat eats a piece of food.
	virtual void PLUGIN_RatConsumedFood() {};
	
 	// Called when the rat moves. "foodFlag" is an int16 field telling
	// if there was food or not at the location where the rat moved. "armNumber"
	// is an int16 field giving the arm number (or CENTER) where the rat moved.
	// This function does not own the message.
	virtual void PLUGIN_RatMoved(BMessage *msg) {};

	///////////////////////////////////////////////////////////////////////////
	// The following EXP_ functions are available to writers of experimenters.
	// These functions define the primary experimenter API.
	///////////////////////////////////////////////////////////////////////////
	  
	// Make the experimenter wait (block) for a number of seconds
	void EXP_Wait(float secs);
	
	// Let the rat do perception or action calls.
	void EXP_PutRatOnMaze();
	void EXP_PutRatOnMaze(int trialType); // Send rat trial type for debugging.
	
	// Stops the rat in his tracks. Blocks him when he tries to do further perception or
	// action calls.
	// duration is the length of time the rat is taken off the maze.
	void EXP_TakeRatOffMaze(float duration=0.0);
	
	// Only valid when the rat is in single step mode. Run rat for a single step.
	// Should only be used in implementation of PLUGIN_RunStep().
	void EXP_SingleStepRat();
	
	void EXP_PlaceFood(int arm);
	void EXP_RemoveFood(int arm);
	void EXP_OpenDoor(int arm);
	void EXP_CloseDoor(int arm);
	
	bool EXP_IsFoodAtLocation(int arm);
	bool EXP_AnyFoodOnMaze(); // food on any arm?
	
	// What direction is the rat facing? (degrees) 0= NORTH, 90= EAST etc.)
	int EXP_GetRatHeading();
	
	// Change position of rat. Arms are 0... MaxArm-1. Center is CENTER (i.e., -1)
	void EXP_PlaceRatAt(int arm);
	
	int EXP_GetNumberOfMazeArms();
	
	// Get the number of rats to be simulated in the current experiment.
	// This is set by the user in Experiment Settings.
	int EXP_GetNumberOfRats();
	
	int EXP_GetCurrentRatNumber();
	void EXP_SetCurrentRatNumber(int ratNo);
	
  	void EXP_AddDatabaseRecord(BMessage *record);

#define		EXP_EVENT_CENTER_ENTRY		1
#define		EXP_EVENT_FOOD_CONSUMED		2
#define		EXP_EVENT_ARM_ENTRY			4
#define		EXP_EVENT_RAT_MOVED 		(EXP_EVENT_CENTER_ENTRY | EXP_EVENT_ARM_ENTRY)
  	
  	// Wait for the rat to do some event, from a set of events. E.g.,
  	// EXP_WaitForEvent(EXP_EVENT_CENTER_ENTRY), or
  	// EXP_WaitForEvent(EXP_EVENT_CENTER_ENTRY | EXP_EVENT_FOOD_CONSUMED).
  	// In the case of waiting for an arm entry, and the arm entry actually occured,
  	// this returns the arm number. In the case of waiting for a center entry, and
  	// the center entry actually occured, this returns CENTER. Return value is
  	// neither CENTER nor a valid arm number when waiting for EXP_EVENT_FOOD_CONSUMED.
  	// If other events occur prior to those specified in the mask, then this
  	// continues to wait and does not return.
	int EXP_WaitForEvent(int eventMask);
	
	// Save the experimenters settings in the database. See PluginSettingsNames.h
	// for allowable field names & types of fields in the settings. Causes
	// settings to be updated in all relevant parts of the maze simulator also.
	void EXP_SaveSettings(BMessage *msg);
	
	// Let the user know that something has happened. These should be used only
	// very infrequently, otherwise, it will be annoying to the user.
	// Produces a GUI message.
	void EXP_UserMessage(char *msg);
	
  private:
  	BMessage *ExperimenterPlugin::WaitForMessage(unsigned int type);
 	// Deal with incoming messages on expEnvPort
	// get rid of any further incoming messages on the port
	void FlushIncomingMessages();
	// Wait for incoming message. Blocks until message ready.
	// Caller owns returned message.
	BMessage *GetIncomingMessage();	
	bool IsIncomingMessageReady();
	
	friend long ReceiveMessages(void *data); // thread that receives messages from port
	
  protected:
     // members shared with derived classes
     BMessage *settings;
     
  private:
  	 // members not shared with derived classes
     UserEnvMessage *userEnvMsg; // for communicating with environment
     PortMessage *envExpPort;
     PortMessage *expRatPort;
     DatabaseMessage *dbMsg;
     WaitQueue *queue;
     thread_id receiveMessagesThreadID;
};


#endif
